from json import load
from sys import argv
from typing import Any, Callable, NotRequired, TypedDict, cast
from urllib.parse import quote, unquote, urljoin, urlsplit, urlunsplit
from urllib.request import (
    HTTPBasicAuthHandler,
    HTTPPasswordMgrWithPriorAuth,
    build_opener,
)

from xbmc import (
    Actor,
    AudioStreamDetail,
    InfoTagVideo,
    SubtitleStreamDetail,
    VideoStreamDetail,
    executebuiltin
)
from xbmcgui import NOTIFICATION_ERROR, Dialog, ListItem
from xbmcplugin import (
    SORT_METHOD_DURATION,
    SORT_METHOD_EPISODE,
    SORT_METHOD_TITLE,
    SORT_METHOD_TITLE_IGNORE_THE,
    SORT_METHOD_VIDEO_RATING,
    SORT_METHOD_VIDEO_RUNTIME,
    SORT_METHOD_VIDEO_YEAR,
    addDirectoryItem,
    addSortMethod,
    endOfDirectory,
    setContent,
    setPluginCategory,
)
from xbmcaddon import Addon

[pluginUrl, handle] = argv[:2]
handle: int = int(handle)
pluginBaseUrl = urlunsplit(urlsplit(pluginUrl)._replace(path="/"))

addon = Addon("plugin.video.kodino")

username = addon.getSetting("username")
password = addon.getSetting("password")
baseUrl = addon.getSetting("baseUrl")


class AudioFormatData(TypedDict):
    lang: list[str]


class VideoFormatData(TypedDict):
    height: int
    width: int


class VideoFormatContainer(TypedDict):
    audio: NotRequired[AudioFormatData]
    video: VideoFormatData


class VideoData(TypedDict):
    cover: str
    directors: dict[str, str]
    duration: float
    episode: int | None
    format: VideoFormatContainer
    group: NotRequired[str]
    imdb: str
    isSeries: bool
    manifest: NotRequired[str]
    rating: float
    season: int | None
    src: str
    stars: dict[str, str]
    subtitles: NotRequired[dict[str, str]]
    title: str
    year: int


type IndexData = dict[str, VideoData]


kodiUrlNetloc = (
    f"{quote(username, safe='')}:{quote(password, safe='')}@{urlsplit(baseUrl).netloc}"
)

passMgr = HTTPPasswordMgrWithPriorAuth()
passMgr.add_password(None, baseUrl, username, password)
auth = HTTPBasicAuthHandler(password_mgr=passMgr)
opener = build_opener(auth)

if "index" not in globals():
    index = None

if "dialog" not in globals():
    dialog = Dialog()


def callApi(path: str) -> Any:
    with opener.open(urljoin(baseUrl, path)) as resp:
        return load(resp)


def getIndex() -> IndexData:
    global index

    if index is None:
        index = callApi("index.json")

    return index


def makeKodiUrl(url: str) -> str:
    return urlunsplit(
        urlsplit(urljoin(baseUrl, quote(url)))._replace(netloc=kodiUrlNetloc)
    )


def videos(
    content: str,
    filter: Callable[[VideoData], bool] = lambda x: True,
    sorter: Callable[[VideoData], Any] = lambda x: x["title"],
):
    index = getIndex()

    for indexItem in sorted([x for x in index.values() if filter(x)], key=sorter):
        indexItem: VideoData = indexItem

        title = indexItem["title"]
        if indexItem["episode"] is not None:
            title = f"Episode {indexItem['episode']}: {title}"

        item = ListItem(title)
        item.setDateTime(str(indexItem["year"]))

        coverUrl = makeKodiUrl(indexItem["cover"])
        item.setArt({"poster": coverUrl, "thumb": coverUrl})

        videoTag = cast(InfoTagVideo, item.getVideoInfoTag())
        videoTag.setUniqueID(indexItem["imdb"], "imdb")
        videoTag.setRating(indexItem["rating"], type="imdb")
        videoTag.setYear(indexItem["year"])
        videoTag.setDuration(int(indexItem["duration"]))
        videoTag.setDirectors(list(indexItem["directors"].values()))
        videoTag.setCast([Actor(name) for name in indexItem["stars"].values()])

        if indexItem["season"] is not None:
            videoTag.addSeason(indexItem["season"])
            videoTag.setSeason(indexItem["season"])
        if indexItem["episode"] is not None:
            videoTag.setEpisode(indexItem["episode"])

        if indexItem["isSeries"] and "group" in indexItem:
            videoTag.setTvShowTitle(indexItem["group"])

        videoTag.addVideoStream(
            VideoStreamDetail(
                indexItem["format"]["video"]["width"],
                indexItem["format"]["video"]["height"],
                indexItem["duration"],
            )
        )

        if "audio" in indexItem["format"]:
            for lang in indexItem["format"]["audio"]["lang"]:
                videoTag.addAudioStream(AudioStreamDetail(language=lang))

        if "subtitles" in indexItem:
            item.setSubtitles(
                [makeKodiUrl(url) for url in indexItem["subtitles"].values()]
            )
            for lang in indexItem["subtitles"]:
                videoTag.addSubtitleStream(SubtitleStreamDetail(lang))

        url = indexItem["src"]
        if "manifest" in indexItem:
            url = indexItem["manifest"]

        addDirectoryItem(
            handle,
            makeKodiUrl(url),
            item,
        )

    setContent(handle, content)
    addSortMethod(handle, SORT_METHOD_VIDEO_YEAR)
    addSortMethod(
        handle,
        SORT_METHOD_DURATION,
    )
    addSortMethod(
        handle,
        SORT_METHOD_EPISODE,
    )
    addSortMethod(
        handle,
        SORT_METHOD_TITLE,
    )
    addSortMethod(
        handle,
        SORT_METHOD_TITLE_IGNORE_THE,
    )
    addSortMethod(
        handle,
        SORT_METHOD_VIDEO_RATING,
    )
    addSortMethod(
        handle,
        SORT_METHOD_VIDEO_RUNTIME,
    )
    endOfDirectory(handle)


def series():
    index = getIndex()

    for series in sorted(set(x["group"] for x in index.values() if x["isSeries"])):
        addDirectoryItem(
            handle,
            pluginBaseUrl + f"series/{quote(series, safe='')}",
            ListItem(series),
            True,
        )

    setContent(handle, "tvshows")
    endOfDirectory(handle)


def seasons(series):
    index = getIndex()

    for season in sorted(
        set(
            x["season"]
            for x in index.values()
            if x["isSeries"] and "group" in x and x["group"] == series
        )
    ):
        if season is None:
            addDirectoryItem(
                handle,
                pluginBaseUrl + f"series/{quote(series, safe='')}/-",
                ListItem("No season"),
                True,
            )
        else:
            addDirectoryItem(
                handle,
                pluginBaseUrl + f"series/{quote(series, safe='')}/{quote(str(season))}",
                ListItem(f"Season {season}"),
                True,
            )

    setContent(handle, "tvshows")
    setPluginCategory(handle, series)
    endOfDirectory(handle)


def main():
    addDirectoryItem(handle, pluginBaseUrl + "movies", ListItem("Movies"), True)
    addDirectoryItem(handle, pluginBaseUrl + "series", ListItem("Series"), True)
    addDirectoryItem(handle, pluginBaseUrl + "reload", ListItem("Reload index"))

    setPluginCategory(handle, "Nichtkino")
    endOfDirectory(handle)


try:
    match [unquote(x) for x in urlsplit(pluginUrl).path.strip("/").split("/")]:
        case [""]:
            main()

        case ["movies"]:
            videos("movies", lambda item: not item["isSeries"])

        case ["series"]:
            series()

        case ["series", series]:
            seasons(series)

        case ["series", series, season]:
            if season != "-":
                setPluginCategory(handle, f"{series} (Season {season})")
                season = int(season)
                videos(
                    "episodes",
                    lambda item: item["isSeries"]
                    and "group" in item
                    and item["group"] == series
                    and "season" in item
                    and item["season"] == season,
                    lambda x: x["episode"],
                )
            else:
                setPluginCategory(handle, f"{series} (No Season)")
                videos(
                    "episodes",
                    lambda item: item["isSeries"]
                    and "group" in item
                    and item["group"] == series
                    and ("season" not in item or item["season"] is None),
                    lambda x: x["episode"],
                )

        case ["reload"]:
            index = None
            try:
                executebuiltin("ActivateWindow(busydialognocancel)")
                getIndex()
                endOfDirectory(handle)
            finally:
                executebuiltin("Dialog.Close(busydialognocancel)")

        case path:
            dialog.notification(
                "Nichtkino",
                f"Unknown submenu {quote(urlsplit(pluginUrl).path)}",
                NOTIFICATION_ERROR,
            )

except Exception as e:
    dialog.notification("Nichtkino", f"Exception occurred: {e}", NOTIFICATION_ERROR)
    endOfDirectory(handle)
